<?php
/*
Plugin Name: ShopScribe Installer
Plugin URI: https://www.shopscribe.de
Description: Automatischer Installer für ShopScribe.
Version: 1.0.0
Author: Themeart GmbH
Author URI: https://www.themeart.de
*/

if (!defined('ABSPATH')) exit;

class ShopScribe_Installer {

    const OPTION_TOKEN = 'shopscribe_installer_shopurl';
    const MENU_SLUG = 'shopscribe-installer';

    // TODO: Deinen API-Endpoint eintragen
    const API_VALIDATE = 'https://www.shopscribe.de/install';

    public function __construct() {
        add_action('admin_menu', [$this, 'menu']);
        add_action('admin_init', [$this, 'register_setting']);
        add_action('admin_enqueue_scripts', [$this, 'admin_enqueue']);
        add_action('wp_ajax_shopscribe_install', [$this, 'ajax_install']);
    }

    public function menu() {
        add_options_page(
            'ShopScribe Installer',
            'ShopScribe Installer',
            'manage_options',
            'shopscribe-installer',
            [$this, 'page']
        );
    }

    public function register_setting() {
        register_setting('shopscribe_installer', self::OPTION_TOKEN, [
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field'
        ]);
    }

    public function admin_enqueue($hook) {

        if ($hook !== 'settings_page_' . self::MENU_SLUG) {
            return;
        }

        wp_enqueue_script('jquery');

        wp_add_inline_style(
            'wp-admin',
            '.shopscribe-plugin-settings { margin-top: 20px; }
            .shopscribe-metabox-header { margin-bottom: 12px; font-style: italic; }
            .shopscribe-plugin-settings .button-primary {
                background-color: #338955;
                border-color: #338955;
            }
            .shopscribe-plugin-settings .button-primary:hover {
                background-color: #3ea767;
                border-color: #3ea767;
            }

            .shopscribe-plugin-header{
                background:#fff;
                display:flex;
                gap:24px;
                align-items:center;
                margin-bottom:24px;
                padding:24px;
                justify-content:space-between;
            }
            .shopscribe-plugin-logo{ height:35px; }
            .shopscribe-plugin-headline{
                font-size:16px;
                font-weight:500;
                margin:0;
                color:#555;
            }
            .shopscribe-post-container h2{ margin-top:0; }
            .shopscribe-postbox{
                background:#fff;
                padding:24px;
                margin-bottom:24px;
                box-shadow:0 8px 24px rgba(0,0,0,.08);
            }
            '
        );
    }

    public function page() {
        $token = get_option(self::OPTION_TOKEN, '');
        $nonce = wp_create_nonce('shopscribe_install_nonce');
        $ajax  = admin_url('admin-ajax.php');

        ?>
        <div class="wrap shopscribe-plugin-settings">
            <div class="shopscribe-plugin-header">
                <a target="_blank" href="https://www.shopscribe.de">
                    <img
                        src="<?php echo plugin_dir_url(__FILE__) . 'assets/logo.svg'; ?>"
                        alt="ShopScribe"
                        class="shopscribe-plugin-logo"
                    />
                </a>
                <div class="shopscribe-plugin-headline">
                    Installer für Shopscribe.
                </div>
            </div>
            <div class="shopscribe-post-container">
                <div class="shopscribe-postbox">
                    <form method="post" action="options.php">
                        <?php settings_fields('shopscribe_installer'); ?>
                        <table class="form-table">
                            <tr>
                                <th scope="row">Shop-URL eintragen</th>
                                <td>
                                    <input type="text" name="<?php echo self::OPTION_TOKEN; ?>"
                                        value="<?php echo esc_attr($token); ?>" class="regular-text">
                                </td>
                            </tr>
                        </table>
                        <?php submit_button('Speichern'); ?>
                    </form>
                </div>
                <div class="shopscribe-postbox">
                    <button id="shopscribe-install" class="button button-primary">ShopScribe Plugin + Theme installieren</button>
                    <div id="shopscribe-result" style="margin-top:20px;"></div>
                </div>
            </div>
        </div>

        <script>
        (function($){
            $('#shopscribe-install').on('click', function(e){
                e.preventDefault();
                $('#shopscribe-result').html("Wird ausgeführt...");

                $.post("<?php echo $ajax; ?>", {
                    action: "shopscribe_install",
                    nonce: "<?php echo $nonce; ?>"
                }, function(response){
                    if(response.success){
                        $('#shopscribe-result').html("<div class='notice notice-success'><p>" + response.data.message + "</p></div>");
                    } else {
                        $('#shopscribe-result').html("<div class='notice notice-error'><p>" + response.data.message + "</p></div>");
                    }
                });
            });
        })(jQuery);
        </script>

        <?php
    }

    public function ajax_install() {
        if (!current_user_can('install_plugins')) {
            wp_send_json_error(['message' => 'Keine Berechtigung.'], 403);
        }

        check_ajax_referer('shopscribe_install_nonce', 'nonce');

        $token = get_option(self::OPTION_TOKEN, '');

        if (empty($token)) {
            wp_send_json_error(['message' => 'Die Shop-Url fehlt.']);
        }

        // Anfrage an shopscribe.de
        $response = wp_remote_post(self::API_VALIDATE, [
            'timeout' => 5,
            'headers' => ['Content-Type' => 'application/json'],
            'User-Agent'   => 'ShopScribe-Installer',
            'body'    => wp_json_encode([
                'shop_url' => $token,
                'wp_url' => home_url(),
                'php_version'=> PHP_MAJOR_VERSION . '.' . PHP_MINOR_VERSION
            ])
        ]);

        if (is_wp_error($response)) {
            wp_send_json_error(['message' => 'API-Fehler: ' . $response->get_error_message()]);
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (!isset($body['status']) || $body['status'] !== 'valid') {
            wp_send_json_error(['message' => $body['message'] ?? 'Anfrage ungültig.']);
        }

        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/plugin-install.php';
        require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        require_once ABSPATH . 'wp-admin/includes/theme.php';

        /* === Plugin installieren === */

        if (!empty($body['plugin']['download_url'])) {
            $url = esc_url_raw($body['plugin']['download_url']);
            $tmp = download_url($url);

            if (is_wp_error($tmp)) {
                wp_send_json_error(['message' => 'Plugin-Download fehlgeschlagen: ' . $tmp->get_error_message()]);
            }

            $upgrader = new Plugin_Upgrader(new Automatic_Upgrader_Skin());
            $result   = $upgrader->install($tmp);
            @unlink($tmp);

            if (!$result || is_wp_error($result)) {
                wp_send_json_error(['message' => 'Plugin-Installation fehlgeschlagen.']);
            }

            // Plugin aktivieren
            $plugin_file = $upgrader->plugin_info();
            if (empty($plugin_file)) {
                wp_send_json_error([
                    'message' => 'Installiertes Plugin konnte nicht ermittelt werden.'
                ]);
            }

            $activate = activate_plugin($plugin_file);
            if (is_wp_error($activate)) {
                wp_send_json_error([
                    'message' => 'Plugin installiert, aber Aktivierung fehlgeschlagen: ' . $activate->get_error_message()
                ]);
            }

            // einstellunegen setzen
            global $wpdb;

            $table = 'shopscribe_shops';

            $token   = !empty($body['token'])    ? sanitize_text_field($body['token'])    : '';
            $shopUrl = !empty($body['shop_url']) ? esc_url_raw($body['shop_url'])         : '';

            if ($token && $shopUrl) {
                $wpdb->delete($table, ['url' => $shopUrl]);
                $wpdb->insert(
                    $table,
                    [
                        'url'   => $shopUrl,
                        'token' => $token,
                    ],
                    [
                        '%s',
                        '%s',
                    ]
                );
            }

        }

        /* === Theme installieren === */

        if (!empty($body['theme']['download_url'])) {
            $url = esc_url_raw($body['theme']['download_url']);
            $tmp = download_url($url);

            if (is_wp_error($tmp)) {
                wp_send_json_error(['message' => 'Theme-Download fehlgeschlagen: ' . $tmp->get_error_message()]);
            }

            $upgrader = new Theme_Upgrader(new Automatic_Upgrader_Skin());
            $result   = $upgrader->install($tmp);
            @unlink($tmp);

            if (!$result || is_wp_error($result)) {
                wp_send_json_error(['message' => 'Theme-Installation fehlgeschlagen.']);
            }
        }

        wp_send_json_success(['message' => 'Plugin + Theme erfolgreich installiert.']);
    }
}

$scInstaller = new ShopScribe_Installer();
